using System;
using System.Collections;
using System.Text.RegularExpressions;
using System.Data;

using gov.va.med.vbecs.DAL.VAL;
using DivTable = gov.va.med.vbecs.Common.VbecsTables.VamcDivision;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Stanislav Antropov</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/16/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	// <summary>
	// Class encapsulating minimum of information needed to identify VA division: code and name.
	// This information is read-only, it's not modified by the VBECS application.
	// </summary>

	#endregion

	/// <summary>
	///		Class encapsulating minimum of information needed to identify VA division: code and name.
	///		This information is read-only, it's not modified by the VBECS application.
	/// </summary>
	public class DivisionDefinition : BaseBusinessObject
	{
		private string _code;
		private string _name;

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4395"> 
		///		<ExpectedInput>Valid division code and name.</ExpectedInput>
		///		<ExpectedOutput>Valid instantiated object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4396"> 
		///		<ExpectedInput>Null as division code.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4397"> 
		///		<ExpectedInput>Null as division name.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4398"> 
		///		<ExpectedInput>Invalid (empty) division code.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4399"> 
		///		<ExpectedInput>Invalid (too long) division code.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4400"> 
		///		<ExpectedInput>Invalid (too short) division name.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Primary constructor creating an instance 
		/// of the class from supplied parameters.
		/// </summary>
		/// <param name="divisionCode">Division code to initialize object with.</param>
		/// <param name="divisionName">Division name to initialize object with.</param>
		public DivisionDefinition( string divisionCode, string divisionName ) 
			: base()
		{
			SetDivisionCode( divisionCode );
			SetDivisionName( divisionName );

			IsNew = false;
			IsDirty = false;
		}

		/// <summary>
		/// Constructor for derived classes.
		/// </summary>
		protected DivisionDefinition() {}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4401"> 
		///		<ExpectedInput>Valid DataRow.</ExpectedInput>
		///		<ExpectedOutput>Valid instantiated object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4402"> 
		///		<ExpectedInput>Null as source DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4403"> 
		///		<ExpectedInput>Invalid (no columns) DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor creating an instance of a the class by reading needed 
		/// parameters from the supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to read data from.</param>
		public DivisionDefinition( DataRow dtRow ) 
			: base()
		{
			// Everything is validated by the callee
			LoadFromDataRow( dtRow );
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			Common.Utility.RequireNonNullColumns( dtRow, GetRequiredColumns() ); // "dtRow not null" is validated here

			SetDivisionCode( (string)dtRow[ DivTable.DivisionCode ] );
			SetDivisionName( (string)dtRow[ DivTable.DivisionName ] );

			IsNew = false;
			IsDirty = false;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4418"> 
		///		<ExpectedInput>Valid DataRow.</ExpectedInput>
		///		<ExpectedOutput>Division code cell in DataRow is populated with division code from the object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4420"> 
		///		<ExpectedInput>Valid DataRow.</ExpectedInput>
		///		<ExpectedOutput>Division name cell in DataRow is populated with division name from the object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4419"> 
		///		<ExpectedInput>Null as DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4421"> 
		///		<ExpectedInput>DataRow with no cells.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			Common.Utility.RequireColumns( dtRow, GetRequiredColumns() ); // "dtRow not null" is validated here

			dtRow[ DivTable.DivisionCode ] = DivisionCode;
			dtRow[ DivTable.DivisionName ] = DivisionName;

			return dtRow;
		}

		/// <summary>
		/// Sets division code.
		/// </summary>
		/// <param name="divisionCode">Division code to set.</param>
		protected void SetDivisionCode( string divisionCode )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );

			if( !Common.RegularExpressions.DivisionCode().IsMatch( divisionCode ) )
				throw( new BusinessObjectException( Common.StrRes.SysErrMsg.Common.InvalidDivisionCodeSupplied().ResString ) );

			_code = divisionCode.Trim().ToUpper();
		}

		/// <summary>
		/// Sets division name.
		/// </summary>
		/// <param name="divisionName">Division name to set.</param>
		protected void SetDivisionName( string divisionName )
		{
			if( divisionName == null )
				throw( new ArgumentNullException( "divisionName" ) );

			if( !Common.RegularExpressions.DivisionName().IsMatch( divisionName ) )
				throw( new BusinessObjectException( Common.StrRes.SysErrMsg.Common.InvalidDivisionNameSupplied().ResString ) );

			_name = divisionName.Trim();
		}

		private static string[] GetRequiredColumns()
		{
			return new string[] { DivTable.DivisionCode, DivTable.DivisionName };
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4409"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Division code (initialized through constructor parameter).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4410"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Division code (initialized through DataRow).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4411"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Division code identifying division within VA.
		/// </summary>
		public virtual string DivisionCode
		{
			get
			{
				return _code;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4412"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Division name (initialized through constructor parameter).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4413"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Division name (initialized through DataRow).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4414"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Division (medical center) name.
		/// </summary>
		public virtual string DivisionName
		{
			get
			{
				return _name;
			}
			set
			{
				this.IsDirty = true;
				_name = value;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4415"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null list.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4416"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-zero count of divisions.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4417"> 
		///		<ExpectedInput>None. VistALink uninitialized.</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves list of VistA divisions via VistALink.
		/// </summary>
		/// <returns>List of <see cref="DivisionDefinition"/> objects representing VistA divisions.</returns>
		public static IList GetDivisionDefListFromVistA()
		{			
			return GetDivisionDefsIListFromDataTable( DAL.VAL.Divisions.GetDivisionList() );
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/27/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8504"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>List of vista institutiions</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8505"> 
		///		<ExpectedInput>None VistALink uninitialized.</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a datatable of vista institutiions, CR 2316
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetDivisionDefTableFromVistA()
		{			
			return 	DAL.VAL.Divisions.GetDivisionList();
		}
	
		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4821"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>List of all VBECS divisions, no exception is thrown.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4822"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets list of all division definitions available in VBECS database.
		/// </summary>
		/// <returns>List of all division definitions in VBECS DB.</returns>
		public static IList GetAllDivisionsDefListFromVbecs()
		{
			return GetDivisionDefsIListFromDataTable( DAL.Division.GetDivisions( false ) );
		}
						
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/23/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7180"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>List of active VBECS divisions, no exception is thrown.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7181"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets list of active division definitions available in VBECS database.
		/// Should be used for all kinds of lookup lists.
		/// </summary>
		/// <returns>List of active division definitions in VBECS DB.</returns>
		public static IList GetActiveVbecsDivisionsDefList()
		{
			return GetDivisionDefsIListFromDataTable( DAL.Division.GetDivisions( true ) );
		}

		/// <summary>
		/// Converts supplied <see cref="DataTable"/> containing division data into 
		/// a list containing corresponding <see cref="DivisionDefinition"/> objects.
		/// </summary>
		/// <param name="dtSource">Source <see cref="DataTable"/></param>.
		/// <returns>List containing corresponding <see cref="DivisionDefinition"/> objects.</returns>
		private static IList GetDivisionDefsIListFromDataTable( DataTable dtSource )
		{
			if( dtSource == null )
				throw( new ArgumentNullException( "dtSource" ) );

			ArrayList _result = new ArrayList( dtSource.Rows.Count );
			
			foreach( DataRow _dr in dtSource.Rows )
				_result.Add( new DivisionDefinition( _dr ) );

			return _result;
		}
	}
}